import math

def f1(x):
    return math.exp(-x * x)

def f2(x):
    return math.sin(x) / x if x != 0 else 1

# 自适应辛普森方法
def adaptive_simpson(f, a, b, tol, max_recursion=50):
    
    def simpson(f, a, b):
        #计算区间 [a, b] 上的辛普森积分
        mid = (a + b) / 2
        return (b - a) / 6 * (f(a) + 4 * f(mid) + f(b))

    def recursive_simpson(f, a, b, tol, depth):
        #递归计算自适应辛普森积分
        mid = (a + b) / 2
        S = simpson(f, a, b)
        S_left = simpson(f, a, mid)
        S_right = simpson(f, mid, b)
        if depth > max_recursion:
            raise RecursionError("达到最大递归深度，可能积分精度不足")
        # 误差估计
        if abs(S_left + S_right - S) < 15 * tol:
            return S_left + S_right + (S_left + S_right - S) / 15
        else:
            # 误差超标，细分区间递归计算
            return (recursive_simpson(f, a, mid, tol / 2, depth + 1) +
                    recursive_simpson(f, mid, b, tol / 2, depth + 1))

    return recursive_simpson(f, a, b, tol, depth=0)

if __name__ == "__main__":
    tolerance = 1e-6  # 容许误差
    a, b = 0, 1       # 积分区间

    result_f1 = adaptive_simpson(f1, a, b, tolerance)
    print(f"f1(x) = exp(-x^2) 在区间 [{a}, {b}] 上的积分结果: {result_f1:.8f}")

    result_f2 = adaptive_simpson(f2, a, b, tolerance)
    print(f"f2(x) = sin(x)/x 在区间 [{a}, {b}] 上的积分结果: {result_f2:.8f}")

